<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Compra;
use App\Models\Producto;
use App\Models\Ubicacion;
use Illuminate\Support\Facades\Auth;

class CompraController extends Controller
{
    public function index(Request $request)
    {
        $query = Compra::with('producto', 'ubicacion', 'usuario');
    
        // Filtro por búsqueda (nombre, código1, código2, descripción)
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->whereHas('producto', function ($q) use ($search) {
                $q->where('nombre', 'like', '%' . $search . '%')
                  ->orWhere('codigo1', 'like', '%' . $search . '%')
                  ->orWhere('codigo2', 'like', '%' . $search . '%')
                  ->orWhere('descripcion', 'like', '%' . $search . '%');
            });
        }
    
        // Filtro por ubicación
        if ($request->filled('ubicacion_id')) {
            $query->where('ubicacion_id', $request->ubicacion_id);
        }
    
        // Filtro por período de fechas
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('fecha_hora', [
                $request->start_date . ' 00:00:00',
                $request->end_date . ' 23:59:59'
            ]);
        }
    
        // Ordenar por fecha en orden descendente (última compra primero)
        $query->orderBy('fecha_hora', 'desc'); // Asumiendo que 'fecha_hora' es el campo de fecha
    
        $ubicaciones = Ubicacion::all();
        $compras = $query->paginate(10)->appends($request->all());
    
        return view('compras.index', compact('compras', 'ubicaciones'));
    }
    
     
    public function create()
    {
        $productos = Producto::all();
        $ubicaciones = Ubicacion::all();

        return view('compras.create', compact('productos', 'ubicaciones'));
    }
    public function store(Request $request)
    {
        $request->validate([
            'producto_id' => 'required|exists:productos,id',
            'ubicacion_id' => 'required|exists:ubicaciones,id',
            'cantidad' => 'required|integer|min:1',
            'costo_total' => 'required|numeric|min:0',
            'observacion' => 'nullable|string|max:255',
        ]);
    
        // Obtener el producto base
        $productoBase = Producto::findOrFail($request->producto_id);
    
        // Verificar si ya existe un producto con el mismo código1 en la misma ubicación
        $productoExistente = Producto::where('codigo1', $productoBase->codigo1)
            ->where('ubicacion_id', $request->ubicacion_id)
            ->first();
    
        if ($productoExistente) {
            // Actualizar la cantidad si ya existe en la ubicación
            $productoExistente->update([
                'cantidad' => $productoExistente->cantidad + $request->cantidad,
            ]);
        } else {
            // Crear una copia del producto en la nueva ubicación con todos los campos relevantes
            Producto::create([
                'nombre' => $productoBase->nombre,
                'codigo1' => $productoBase->codigo1,
                'codigo2' => $productoBase->codigo2,
                'descripcion' => $productoBase->descripcion,
                'precio_unidad_facturado' => $productoBase->precio_unidad_facturado,
                'precio_docena_facturado' => $productoBase->precio_docena_facturado,
                'precio_paquete_facturado' => $productoBase->precio_paquete_facturado,
                'precio_paquete_normal' => $productoBase->precio_paquete_normal,
                'precio_docena_normal' => $productoBase->precio_docena_normal,
                'precio_docena_caja' => $productoBase->precio_docena_caja,
                'piezas_por_paquete' => $productoBase->piezas_por_paquete,
                'piezas_por_caja' => $productoBase->piezas_por_caja,
                'ubicacion_deposito' => $productoBase->ubicacion_deposito,
                'descuento' => $productoBase->descuento,
                'incremento' => $productoBase->incremento,
                'marca' => $productoBase->marca,
                'foto' => $productoBase->foto,
                'cantidad' => $request->cantidad,
                'ubicacion_id' => $request->ubicacion_id,
                'estado' => 'activo',
                'posicion' => $productoBase->posicion,
                'docena' => $productoBase->docena,
            ]);
        }
    
        // Registrar la compra
        Compra::create([
            'user_id' => Auth::id(),
            'producto_id' => $request->producto_id,
            'ubicacion_id' => $request->ubicacion_id,
            'cantidad' => $request->cantidad,
            'costo_total' => $request->costo_total,
            'fecha_hora' => now(),
            'observacion' => $request->observacion,
        ]);
    
        return redirect()->route('compras.index')->with('success', 'Compra registrada exitosamente.');
    }
    


    public function generatePDF(Request $request)
{
    $query = Compra::with('producto', 'ubicacion', 'usuario');

    // Filtro por nombre de producto
    if ($request->filled('search')) {
        $query->whereHas('producto', function ($q) use ($request) {
            $q->where('nombre', 'like', '%' . $request->search . '%');
        });
    }

    // Filtro por ubicación
    if ($request->filled('ubicacion_id')) {
        $query->where('ubicacion_id', $request->ubicacion_id);
    }

    // Filtro por período de fechas
    if ($request->filled('start_date') && $request->filled('end_date')) {
        $query->whereBetween('fecha_hora', [
            $request->start_date . ' 00:00:00',
            $request->end_date . ' 23:59:59'
        ]);
    }

    $compras = $query->get();

    // Crear el título del PDF con fecha y hora actual
    $pdfTitle = 'reporte_compras_' . now()->format('d_m_Y_H_i_s');

    // Generar el PDF usando DomPDF
    $pdf = \PDF::loadView('compras.pdf', compact('compras', 'pdfTitle'));

    return $pdf->download($pdfTitle . '.pdf');
}

}
