<?php

namespace App\Http\Controllers;

use App\Models\Empleado;
use App\Models\Ubicacion;
use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;

class EmpleadoController extends Controller
{
    public function index(Request $request)
    {
        $query = Empleado::query()->with('sucursal');

        // Filtro por sucursal
        if ($request->filled('sucursal_id')) {
            $query->where('ubicacion_id', $request->sucursal_id);
        }

        // Búsqueda por carnet, nombre o apellidos
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('carnet', 'like', "%$search%")
                    ->orWhere('nombre', 'like', "%$search%")
                    ->orWhere('apellidos', 'like', "%$search%");
            });
        }

        // Paginación
        $empleados = $query->paginate(5)->appends($request->all());

        // Obtener sucursales para el filtro
        $ubicaciones = Ubicacion::all();

        return view('empleados.index', compact('empleados', 'ubicaciones'));
    }

    public function create()
    {
        $ubicaciones = Ubicacion::all();
        return view('empleados.create', compact('ubicaciones'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ\'\s]+$/u'
            ],
            'apellidos' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ\'\s]+$/u'
            ],
            'carnet' => 'required|digits_between:6,15|unique:empleados',
            'fecha_nacimiento' => 'required|date|before:' . now()->subYears(18)->format('Y-m-d'),
            'direccion' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ0-9#.,\s]+$/u'
            ],
            'telefono' => 'required|digits:8',
            'correo' => 'required|email|unique:empleados',
            'ubicacion_id' => 'required|exists:ubicaciones,id',
        ], [
            'nombre.required' => 'El nombre es obligatorio.',
            'nombre.regex' => 'El nombre solo puede contener letras, espacios y caracteres válidos.',
            'apellidos.required' => 'Los apellidos son obligatorios.',
            'apellidos.regex' => 'Los apellidos solo pueden contener letras, espacios y caracteres válidos.',
            'carnet.required' => 'El carnet es obligatorio.',
            'carnet.digits_between' => 'El carnet debe tener entre 6 y 15 números.',
            'carnet.unique' => 'El carnet ya está registrado.',
            'fecha_nacimiento.required' => 'La fecha de nacimiento es obligatoria.',
            'fecha_nacimiento.date' => 'La fecha de nacimiento debe ser válida.',
            'fecha_nacimiento.before' => 'Debe ser mayor de 18 años.',
            'direccion.required' => 'La dirección es obligatoria.',
            'direccion.regex' => 'La dirección solo puede contener letras, números y caracteres válidos.',
            'telefono.required' => 'El teléfono es obligatorio.',
            'telefono.digits' => 'El teléfono debe tener 8 dígitos.',
            'correo.required' => 'El correo es obligatorio.',
            'correo.email' => 'El correo debe ser válido.',
            'correo.unique' => 'El correo ya está registrado.',
            'ubicacion_id.required' => 'Debe seleccionar una sucursal.',
            'ubicacion_id.exists' => 'La sucursal seleccionada no es válida.',
        ]);

        Empleado::create($request->all());
        return redirect()->route('empleados.index')->with('success', 'Empleado creado correctamente.');
    }

    public function edit(Empleado $empleado)
    {
        $ubicaciones = Ubicacion::all();
        return view('empleados.edit', compact('empleado', 'ubicaciones'));
    }

    public function update(Request $request, Empleado $empleado)
    {
        $request->validate([
            'nombre' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ\'\s]+$/u'
            ],
            'apellidos' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ\'\s]+$/u'
            ],
            'carnet' => 'required|digits_between:6,15|unique:empleados,carnet,' . $empleado->id,
            'fecha_nacimiento' => 'required|date|before:' . now()->subYears(18)->format('Y-m-d'),
            'direccion' => [
                'required',
                'regex:/^[a-zA-ZÁÉÍÓÚÜÑáéíóúüñ0-9#.,\s]+$/u'
            ],
            'telefono' => 'required|digits:8',
            'correo' => 'required|email|unique:empleados,correo,' . $empleado->id,
            'ubicacion_id' => 'required|exists:ubicaciones,id',
        ], [
            // Mensajes personalizados (idénticos a los del método store)
        ]);

        $empleado->update($request->all());
        return redirect()->route('empleados.index')->with('success', 'Empleado actualizado correctamente.');
    }

    public function destroy(Empleado $empleado)
    {
        // Cambiar el estado: si es 1, se cambia a 0; si es 0, se cambia a 1
        $empleado->estado = $empleado->estado == 1 ? 0 : 1;
        $empleado->save();
    
        // Mensaje dinámico según el nuevo estado
        $message = $empleado->estado == 1 
            ? 'Empleado activado correctamente.' 
            : 'Empleado desactivado correctamente.';
    
        return redirect()->route('empleados.index')->with('success', $message);
    }
    
    public function generateUserForm($empleadoId)
    {
        $empleado = Empleado::find($empleadoId);
    
        // Verificar si existe
        if (!$empleado) {
            return redirect()->route('empleados.index')->with('error', 'Empleado no encontrado.');
        }
    
        return view('empleados.generate_user', compact('empleado'));
    }
    
    public function generateUser(Request $request, $empleadoId)
    {
        $empleado = Empleado::find($empleadoId);
    
        // Validar la contraseña
        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);
    
        try {
            // Crear un nuevo usuario
            $user = new User([
                'name' => $empleado->nombre,
                'email' => $empleado->correo,
                'password' => bcrypt($request->password),
            ]);
    
            // Guardar el nuevo usuario
            $user->save();
    
            // Asociar el usuario con el empleado
            $empleado->user()->associate($user);
            $empleado->save();
    
            // Asignar el rol al usuario
            $roleName = 'Empleado';
            $role = Role::where('name', $roleName)->first();
            if (!$role) {
                $role = Role::create(['name' => $roleName]);
            }
            $user->assignRole($roleName);
    
            return redirect()->route('empleados.index')->with('success', 'Usuario generado correctamente para ' . $empleado->nombre);
        } catch (QueryException $e) {
            // Capturar el error de duplicación de correo
            if ($e->errorInfo[1] == 1062) { // Código de error 1062 para entradas duplicadas
                $customErrorMessage = 'El correo electrónico ya está registrado en el sistema. Por favor, utiliza otro correo.';
            } else {
                $customErrorMessage = 'Error inesperado al crear el usuario. Por favor, inténtalo nuevamente.';
            }
    
            return redirect()->back()->with('error', $customErrorMessage);
        } catch (\Exception $e) {
            // Manejo general de excepciones
            return redirect()->back()->with('error', 'El correo electrónico ya está registrado en el sistema. Por favor, utiliza otro correo.');
        }
    }
    
    
}
