<?php

namespace App\Http\Controllers;

use App\Models\Producto;
use App\Models\Ubicacion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\QueryException;
use App\Imports\ImportProductos;
use App\Models\ArchivoImportado;
use App\Models\TraspasoConfig;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf as DomPDFFacadePdf;
use Dompdf\Options;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Picqer\Barcode\BarcodeGeneratorPNG;
use Illuminate\Validation\ValidationException;
class ProductoController extends Controller
{
    public function index(Request $request)
    {
        $query = Producto::query();
    
        // Filtrar por estado si no es administrador
        if (!auth()->user()->hasRole('Super administrador')) {
            $query->where('estado', 'activo');
        }
    
        // Filtro por búsqueda
        if ($request->has('q') && !empty($request->input('q'))) {
            $q = $request->input('q');
            $query->where(function ($subQuery) use ($q) {
                $subQuery->where('nombre', 'like', "%$q%")
                    ->orWhere('descripcion', 'like', "%$q%")
                    ->orWhere('codigo1', 'like', "%$q%")
                    ->orWhere('codigo2', 'like', "%$q%");
            });
        }
    
        // Filtro por ubicación
        if ($request->has('ubicacion_id') && !empty($request->input('ubicacion_id'))) {
            $ubicacionId = $request->input('ubicacion_id');
            $query->where('ubicacion_id', $ubicacionId);
        }
    
        // Ordenar por fecha de creación descendente (últimos productos primero)
        $query->orderBy('created_at', 'desc');
    
        // Obtener ubicaciones y productos
        $ubicaciones = Ubicacion::all();
        $productos = $query->paginate(10)->appends($request->all());
    
        // Obtener tipo de cambio
        $tipoCambioConfig = TraspasoConfig::first();
        $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1;
    
        // Aplicar tipo de cambio a los precios
        foreach ($productos as $producto) {
            $producto->precio_unidad_facturado *= $tipoCambio;
            $producto->precio_docena_facturado *= $tipoCambio;
            $producto->precio_paquete_facturado *= $tipoCambio;
            $producto->precio_paquete_normal *= $tipoCambio;
            $producto->precio_docena_normal *= $tipoCambio;
            $producto->precio_docena_caja *= $tipoCambio;
        }
    
        return view('mercancia.index', compact('productos', 'tipoCambio', 'ubicaciones'));
    }
    
    public function actualizarTipoCambio(Request $request)
    {
        $request->validate([
            'tipo_cambio' => 'required|numeric|min:0|regex:/^\d+(\.\d{1,2})?$/',
        ]);
    
        $tipoCambioConfig = TraspasoConfig::first();
        if ($tipoCambioConfig) {
            $tipoCambioConfig->update(['tipo_cambio' => $request->input('tipo_cambio')]);
        } else {
            TraspasoConfig::create(['tipo_cambio' => $request->input('tipo_cambio')]);
        }
    
        return back()->with('success', 'Tipo de cambio aplicado correctamente.');
    }
    


    // Método para eliminar (reiniciar) el tipo de cambio
    public function eliminarTipoCambio()
    {
        // Reiniciar el tipo de cambio a 1 en la tabla TraspasoConfig
        $tipoCambioConfig = TraspasoConfig::first();
        if ($tipoCambioConfig) {
            $tipoCambioConfig->update(['tipo_cambio' => 1]);
        }

        return redirect()->route('mercancia.index')->with('success', 'Tipo de cambio reiniciado a 1.');
    }

public function search(Request $request)
{
    try {
        $q = $request->input('q');

        $productos = Producto::where('nombre', 'like', "%$q%")
            ->orWhere('codigo1', 'like', "%$q%")
            ->orWhere('descripcion', 'like', "%$q%")
            ->orWhere('codigo2', 'like', "%$q%")
            ->get();

        // Verifica si no hay resultados y devuelve un mensaje adecuado
        if ($productos->isEmpty()) {
            return response()->json(['error' => 'No se encontraron productos'], 404);
        }

        // Renderiza la sección de la tabla
        $tableHtml = view('mercancia.index_table', compact('productos'))->render();

        return response()->json(['tableHtml' => $tableHtml]);
    } catch (\Exception $e) {
        // Maneja cualquier error de forma genérica
        return response()->json(['error' => $e->getMessage()], 500);
    }
}

public function importar(Request $request)
{
    $request->validate([
        'file' => 'required|mimes:xlsx,csv,xls',
    ]);

    $file = $request->file('file');
    $fileName = $file->getClientOriginalName();
    $filePath = $file->storeAs('importaciones', time() . '_' . $fileName, 'public');

    // Registrar el archivo en la base de datos
    ArchivoImportado::create([
        'nombre' => $fileName,
        'ruta' => $filePath,
    ]);

    // Leer el archivo Excel
    $spreadsheet = IOFactory::load($file->getRealPath());
    $rows = $spreadsheet->getActiveSheet()->toArray();
    array_shift($rows); // Eliminar la primera fila (encabezados)

    foreach ($rows as $row) {
        // Validar si la fila está vacía, verificando los campos clave
        if (empty($row[0]) && empty($row[1]) && empty($row[16])) {
            continue; // Omitir la fila vacía
        }

        $nombre = $row[0] ?? 'Sin nombre';
        $codigo1 = $row[1] ?? null;
        $codigo2 = $row[2] ?? null;
        $descripcion = $row[3] ?? 'Sin descripción';

        $precios = [
            'precio_unidad_facturado' => $row[4] ?? 0,
            'precio_docena_facturado' => $row[5] ?? 0,
            'precio_paquete_facturado' => $row[6] ?? 0,
            'precio_paquete_normal' => $row[7] ?? 0,
            'precio_docena_normal' => $row[8] ?? 0,
            'precio_docena_caja' => $row[9] ?? 0,
        ];

        $descuento = $row[10] ?? 0;
        $incremento = $row[11] ?? 0;

        // Validar que no haya incremento y descuento simultáneamente
        if ($incremento > 0 && $descuento > 0) {
            continue;
        }

        $marca = $row[12] ?? 'Sin marca';
        $foto = $row[13] ?? null;
        $piezasPorPaquete = $row[14] ?? 1;
        $piezasPorCaja = $row[15] ?? 1;
        $ubicacionNombre = $row[16] ?? 'Ubicación Desconocida';
        $cantidad = $row[17] ?? 0;
        $posicion = $row[18] ?? 'Sin posición';
        $docena = isset($row[19]) && strtolower($row[19]) === 'si' ? true : false; // Nuevo campo
        if (is_null($nombre) || is_null($codigo1) || is_null($ubicacionNombre)) {
            throw ValidationException::withMessages([
                'file' => 'El archivo no cuenta con el formato o la ubicación requerida, revise los campos: Nombre, Código1, y Ubicación.',
            ]);
        }

        // Obtener o crear la ubicación
        $ubicacion = Ubicacion::firstOrCreate(['nombre' => $ubicacionNombre]);

        // Aplicar incremento o descuento
        foreach ($precios as $key => &$value) {
            if ($incremento > 0) {
                $value += ($value * $incremento / 100);
            } elseif ($descuento > 0) {
                $value -= ($value * $descuento / 100);
            }
        }

        // Buscar el producto por código1 y código2
        $producto = Producto::where('codigo1', $codigo1)
                            ->where('codigo2', $codigo2)
                            ->first();

        if ($producto) {
            // Si el producto existe, actualizar la cantidad
            $producto->cantidad += $cantidad;
            $producto->save();
        } else {
            // Si el producto no existe, crear uno nuevo
            Producto::create([
                'nombre' => $nombre,
                'codigo1' => $codigo1,
                'codigo2' => $codigo2,
                'descripcion' => $descripcion,
                'marca' => $marca,
                'foto' => $foto,
                'cantidad' => $cantidad,
                'piezas_por_paquete' => $piezasPorPaquete,
                'piezas_por_caja' => $piezasPorCaja,
                'ubicacion_id' => $ubicacion->id,
                'estado' => 'activo',
                'descuento' => $descuento,
                'incremento' => $incremento,
                'posicion' => $posicion,
                'docena' => $docena, // Nuevo campo
            ] + $precios);
        }
    }

    return redirect()->route('mercancia.index')->with('success', 'Productos importados correctamente.');
}

public function gestionarArchivos()
{
    $archivos = ArchivoImportado::orderBy('created_at', 'desc')->paginate(10);
    return view('archivos.index', compact('archivos'));
}
public function descargarArchivo($archivoId)
{
    $archivo = ArchivoImportado::findOrFail($archivoId);
    $rutaArchivo = storage_path('app/public/' . $archivo->ruta);

    if (file_exists($rutaArchivo)) {
        return response()->download($rutaArchivo, $archivo->nombre);
    }

    return redirect()->route('archivos.index')->with('error', 'El archivo no existe en el servidor.');
}

public function eliminarArchivo($archivoId)
{
    try {
        $archivo = ArchivoImportado::findOrFail($archivoId);

        // Eliminar productos relacionados al archivo
        Producto::where('created_at', '>=', $archivo->created_at)->where('created_at', '<=', now())->delete();

        // Eliminar el archivo físico
        Storage::delete($archivo->ruta);

        // Eliminar registro del archivo
        $archivo->delete();

        return redirect()->route('archivos.index')->with('success', 'Archivo y productos asociados eliminados correctamente.');
    } catch (\Exception $e) {
        return redirect()->route('archivos.index')->with('error', 'Error al eliminar archivo: ' . $e->getMessage());
    }
}

public function create()
    {
        $ubicaciones = Ubicacion::all();
        return view('mercancia.create', compact('ubicaciones'));
    }
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'codigo1' => 'nullable|regex:/^[a-zA-Z0-9ñÑ\/\.\-_]+$/',
            'codigo2' => 'nullable|regex:/^[a-zA-Z0-9ñÑ\/\.\-_]+$/',
            'descripcion' => 'required|string|max:255',
            'posicion' => 'nullable|string|max:100',
            'docena' => 'required|boolean',
            'precio_unidad_facturado' => 'required|numeric|min:0',
            'precio_docena_facturado' => 'nullable|numeric|min:0',
            'precio_paquete_facturado' => 'nullable|numeric|min:0',
            'precio_paquete_normal' => 'nullable|numeric|min:0',
            'precio_docena_normal' => 'nullable|numeric|min:0',
            'precio_docena_caja' => 'nullable|numeric|min:0',
            'descuento' => 'nullable|numeric|min:0|max:100',
            'incremento' => 'nullable|numeric|min:0|max:100',
            'piezas_por_paquete' => 'required|integer|min:1',
            'piezas_por_caja' => 'required|integer|min:1',
            'ubicacion_id' => 'required|exists:ubicaciones,id',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'marca' => 'nullable|string|max:255',
            'cantidad' => 'nullable|integer|min:0',
        ]);
    
        if ($request->incremento > 0 && $request->descuento > 0) {
            return redirect()->back()->with('error', 'No se puede aplicar incremento y descuento al mismo tiempo.');
        }
    
        $rutaFoto = $request->hasFile('foto') ? $request->file('foto')->store('productos', 'public') : null;
    
        // Datos base
        $data = $request->only([
            'nombre',
            'codigo1',
            'codigo2',
            'descripcion',
            'posicion',
            'piezas_por_paquete',
            'piezas_por_caja',
            'ubicacion_id',
            'marca',
            'cantidad',
            'descuento',
            'incremento',
        ]);
        $data['foto'] = $rutaFoto;
    
        // Aplicar incremento o descuento dinámicamente
        foreach ([
            'precio_unidad_facturado',
            'precio_docena_facturado',
            'precio_paquete_facturado',
            'precio_paquete_normal',
            'precio_docena_normal',
            'precio_docena_caja',
        ] as $key) {
            $value = $request->$key;
    
            if ($request->incremento > 0) {
                $data[$key] = $value + ($value * $request->incremento / 100);
            } elseif ($request->descuento > 0) {
                $data[$key] = $value - ($value * $request->descuento / 100);
            } else {
                $data[$key] = $value; // Mantener precio ingresado
            }
        }
    
        Producto::create($data);
    
        return redirect()->route('mercancia.index')->with('success', 'Producto creado exitosamente.');
    }
    
    

    
public function cambiarEstado( Producto $producto)
{
  

    $nuevoEstado = ($producto->estado === 'activo') ? 'inactivo' : 'activo';
    $producto->update(['estado' => $nuevoEstado]);

    return redirect()->route('mercancia.index')->with('success', 'Estado del producto cambiado exitosamente.');
}





    

    public function show(Producto $producto)
    {
        return view('mercancia.show', compact('producto'));
    }

    public function edit(Producto $producto)
    {
        $ubicaciones = Ubicacion::all();
        return view('mercancia.edit', compact('producto', 'ubicaciones'));
    }
   

    public function update(Request $request, Producto $producto)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'codigo1' => 'nullable|regex:/^[a-zA-Z0-9ñÑ\/\.\-_]+$/',
            'codigo2' => 'nullable|regex:/^[a-zA-Z0-9ñÑ\/\.\-_]+$/',
            'descripcion' => 'required|string|max:1000',
            'posicion' => 'nullable|string|max:100',
            'docena' => 'required|boolean',
            'precio_unidad_facturado' => 'required|numeric|min:0',
            'precio_docena_facturado' => 'nullable|numeric|min:0',
            'precio_paquete_facturado' => 'nullable|numeric|min:0',
            'precio_paquete_normal' => 'nullable|numeric|min:0',
            'precio_docena_normal' => 'nullable|numeric|min:0',
            'precio_docena_caja' => 'nullable|numeric|min:0',
            'marca' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'ubicacion_id' => 'required|exists:ubicaciones,id',
            'cantidad' => 'required|integer|min:0',
            'incremento' => 'nullable|numeric|min:0|max:100',
            'descuento' => 'nullable|numeric|min:0|max:100',
            'piezas_por_paquete' => 'required|integer|min:1',
            'piezas_por_caja' => 'required|integer|min:1',
        ]);
    
        if ($request->incremento > 0 && $request->descuento > 0) {
            return redirect()->back()->with('error', 'No se puede aplicar incremento y descuento al mismo tiempo.');
        }
    
        // Manejo de foto
        $rutaFoto = $producto->foto;
        if ($request->hasFile('foto')) {
            if ($producto->foto) {
                Storage::delete('public/' . $producto->foto);
            }
            $rutaFoto = $request->file('foto')->store('productos', 'public');
        }
    
        // Extraer precios del formulario
        $preciosActualizados = $request->only([
            'precio_unidad_facturado',
            'precio_docena_facturado',
            'precio_paquete_facturado',
            'precio_paquete_normal',
            'precio_docena_normal',
            'precio_docena_caja',
        ]);
    
        // Revertir incrementos/descuentos previos si aplican
        if ($producto->incremento > 0) {
            foreach ($preciosActualizados as $key => $value) {
                $preciosActualizados[$key] = $value / (1 + $producto->incremento / 100);
            }
        } elseif ($producto->descuento > 0) {
            foreach ($preciosActualizados as $key => $value) {
                $preciosActualizados[$key] = $value / (1 - $producto->descuento / 100);
            }
        }
    
        // Aplicar nuevos incrementos/descuentos si se establecen
        if ($request->incremento > 0) {
            foreach ($preciosActualizados as $key => $value) {
                $preciosActualizados[$key] = $value + ($value * $request->incremento / 100);
            }
        } elseif ($request->descuento > 0) {
            foreach ($preciosActualizados as $key => $value) {
                $preciosActualizados[$key] = $value - ($value * $request->descuento / 100);
            }
        }
    
        // Combinar datos actualizados con otros valores
        $updatedData = $request->only([
            'nombre',
            'codigo1',
            'codigo2',
            'descripcion',
            'posicion',
            'docena',
            'piezas_por_paquete',
            'piezas_por_caja',
            'ubicacion_id',
            'marca',
            'cantidad',
            'incremento',
            'descuento',
        ]);
        $updatedData['foto'] = $rutaFoto;
    
        // Mezclar precios actualizados
        $updatedData = array_merge($updatedData, $preciosActualizados);
    
        // Actualizar el producto
        $producto->update($updatedData);
    
        return redirect()->route('mercancia.index')->with('success', 'Producto actualizado exitosamente.');
    }
    
    
    public function generarReportePDF(Request $request)
    {
        $query = Producto::query();
    
        // Filtros
        if ($request->filled('q')) {
            $q = $request->input('q');
            $query->where(function ($subQuery) use ($q) {
                $subQuery->where('nombre', 'like', "%$q%")
                    ->orWhere('descripcion', 'like', "%$q%")
                    ->orWhere('codigo1', 'like', "%$q%")
                    ->orWhere('codigo2', 'like', "%$q%");
            });
        }
    
        if ($request->filled('ubicacion_id')) {
            $ubicacionId = $request->input('ubicacion_id');
            $query->where('ubicacion_id', $ubicacionId);
        }
    
        $productos = $query->get();
    
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('mercancia.reporte_pdf', compact('productos'));
    
        // Obtener la fecha y hora del equipo
        $fechaHora = now()->format('Y_m_d_His');
        $nombreArchivo = "reporte_productos_{$fechaHora}.pdf";
    
        return $pdf->stream($nombreArchivo, ['Attachment' => false]);
    }
    
    
    public function destroy(Producto $producto)
    {
       
        $producto->ubicaciones()->detach();
    
        $producto->delete();
    
        return redirect()->route('mercancia.index')->with('success', 'Producto eliminado exitosamente.');
    }
    
   
    public function generarSticker($id)
    {
        $producto = Producto::findOrFail($id);
    
        // Obtener tipo de cambio desde la configuración
        $tipoCambioConfig = TraspasoConfig::first();
        $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1;
    
    
    
        // Generar código de barras en formato PNG
        $generator = new BarcodeGeneratorPNG();
        $barcode = base64_encode($generator->getBarcode($producto->codigo2, $generator::TYPE_CODE_128));
    
        // Renderiza la vista del sticker con los datos
        $pdf = DomPDFFacadePdf::loadView('mercancia.sticker', compact('producto', 'barcode', 'tipoCambio'))
            ->setPaper([0, 0, 141.732, 141.732], 'portrait'); // Tamaño 5x5 cm en puntos
        
        return $pdf->stream('sticker_producto_' . $producto->codigo1 . '.pdf');
    }
    
    
    public function generarStickerPPN($id)
{
    $producto = Producto::findOrFail($id);

    // Obtener tipo de cambio desde la configuración
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1;

    // Aplicar tipo de cambio a los precios
  

    // Generar código de barras en formato PNG
    $generator = new BarcodeGeneratorPNG();
    $barcode = base64_encode($generator->getBarcode($producto->codigo2, $generator::TYPE_CODE_128));

    // Renderiza la vista del sticker con los datos
    $ppn = 'PPN'; // Reemplaza por PPN
    $ppf = 'PPF'; // Reemplaza por PPF
    $pdf = DomPDFFacadePdf::loadView('mercancia.sticker_ppn', compact('producto', 'barcode', 'tipoCambio', 'ppn', 'ppf'))
        ->setPaper([0, 0, 141.732, 141.732], 'portrait'); // Tamaño 5x5 cm en puntos
    
    return $pdf->stream('sticker_producto_' . $producto->codigo1 . '.ppn');
}

public function generarStickerPPNGrande($id)
{
    $producto = Producto::findOrFail($id);
    
    // Obtener tipo de cambio desde la configuración
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1;
    
    // Generar código de barras en formato PNG
    $generator = new \Picqer\Barcode\BarcodeGeneratorPNG();
    $barcode = base64_encode($generator->getBarcode($producto->codigo2, $generator::TYPE_CODE_128));
    
    // Generar el PDF con orientación horizontal
    $pdf = DomPDFFacadePdf::loadView('mercancia.stickerPPNGrande', [
        'producto' => $producto,
        'barcode' => $barcode,
        'tipoCambio' => $tipoCambio,
    ])
    ->setPaper('a5', 'landscape'); // Usa un tamaño estándar en orientación horizontal

    return $pdf->stream('sticker_producto_paquete_grande_' . $producto->codigo1 . '.pdf');
}



public function generarStickerPUFGrande($id)
{
    $producto = Producto::findOrFail($id);
    
    // Obtener tipo de cambio desde la configuración
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1;
    
    // Generar código de barras en formato PNG
    $generator = new \Picqer\Barcode\BarcodeGeneratorPNG();
    $barcode = base64_encode($generator->getBarcode($producto->codigo2, $generator::TYPE_CODE_128));
    
    // Generar el PDF con orientación horizontal
    $pdf = DomPDFFacadePdf::loadView('mercancia.stickerPUFGrande', [
        'producto' => $producto,
        'barcode' => $barcode,
        'tipoCambio' => $tipoCambio,
    ])
    ->setPaper('a5', 'landscape'); // Usa un tamaño estándar en horizontal

    return $pdf->stream('sticker_producto_grande_' . $producto->codigo1 . '.pdf');
}




}