<?php


namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Traspaso;
use App\Models\Producto;
use App\Models\Ubicacion;
use Illuminate\Support\Facades\Auth;
use App\Models\TraspasoConfig;
class TraspasoController extends Controller
{
    public function index(Request $request)
    {
        $query = Traspaso::with('ubicacionOrigen', 'ubicacionDestino', 'producto', 'usuario');
    
        // Filtro por producto
        if ($request->has('search') && !empty($request->input('search'))) {
            $search = $request->input('search');
            $query->whereHas('producto', function ($q) use ($search) {
                $q->where('nombre', 'like', "%$search%");
            });
        }
    
        // Filtro por estado
        if ($request->has('estado') && !empty($request->input('estado'))) {
            $query->where('estado', $request->estado);
        }
    
        // Filtro por sucursal de origen
        if ($request->has('ubicacion_origen_id') && !empty($request->ubicacion_origen_id)) {
            $query->where('ubicacion_origen_id', $request->ubicacion_origen_id);
        }
    
        // Filtro por sucursal de destino
        if ($request->has('ubicacion_destino_id') && !empty($request->ubicacion_destino_id)) {
            $query->where('ubicacion_destino_id', $request->ubicacion_destino_id);
        }
    
        $traspasos = $query->paginate(5)->appends(request()->all());
    
        // Obtener todas las ubicaciones para los filtros
        $ubicaciones = Ubicacion::all();
    
        return view('traspasos.index', compact('traspasos', 'ubicaciones'));
    }
    
    public function obtenerProductosPorUbicacion($ubicacionId)
{
    try {
        // Consulta los productos por ubicación y estado activo
        $productos = Producto::where('ubicacion_id', $ubicacionId)
            ->where('estado', 'activo')
            ->get([
                'id',
                'nombre',
                'codigo1',
                'codigo2',
                'descripcion',
                'precio_unidad_facturado',
                'precio_docena_facturado',
                'precio_paquete_facturado',
                'precio_paquete_normal',
                'precio_docena_normal',
                'precio_docena_caja',
                'piezas_por_paquete',
                'piezas_por_caja',
                'ubicacion_deposito',
                'descuento',
                'incremento',
                'marca',
                'cantidad',
                'posicion'
            ]);

        // Validar si hay resultados
        if ($productos->isEmpty()) {
            return response()->json(['error' => 'No se encontraron productos en esta ubicación.'], 404);
        }

        // Retorna los productos en formato JSON
        return response()->json($productos, 200);
    } catch (\Exception $e) {
        // Registrar el error en los logs
        \Log::error('Error al obtener productos: ' . $e->getMessage());

        // Respuesta de error genérica
        return response()->json(['error' => 'Ocurrió un error al obtener los productos.'], 500);
    }
}


    public function showFormEnviarProducto()
    {
        // Obtener todas las ubicaciones disponibles
        $ubicaciones = Ubicacion::all();
    
        // Obtener productos que tengan una ubicación válida
        $productosPorUbicacion = Producto::whereNotNull('ubicacion_id')
            ->with('ubicacion')
            ->get()
            ->groupBy('ubicacion_id');
    
        return view('traspasos.enviar_producto', compact('ubicaciones', 'productosPorUbicacion'));
    }


    public function enviarProducto(Request $request)
    {
        // Obtener configuración de límite máximo
        $configuracion = TraspasoConfig::first();
        $cantidadMaxima = $configuracion ? $configuracion->cantidad_maxima : 0;
    
        $request->validate([
            'producto_id' => 'required|exists:productos,id',
            'ubicacion_origen_id' => 'required|exists:ubicaciones,id',
            'ubicacion_destino_id' => 'required|exists:ubicaciones,id|different:ubicacion_origen_id',
            'cantidad' => 'required|integer|min:1|max:' . $cantidadMaxima,
        ], [
            'ubicacion_destino_id.different' => 'La sucursal de destino debe ser diferente a la sucursal de origen.',
            'cantidad.max' => "La cantidad no puede ser mayor a {$cantidadMaxima}.",
        ]);
    
        // Buscar el producto en la sucursal de origen
        $producto = Producto::where('id', $request->producto_id)
            ->where('ubicacion_id', $request->ubicacion_origen_id)
            ->firstOrFail();
    
        // Validar cantidad disponible en la sucursal de origen
        if ($producto->cantidad < $request->cantidad) {
            return redirect()->back()->with('error', 'Cantidad insuficiente en la sucursal de origen.');
        }
    
        // Reducir la cantidad del producto en la sucursal de origen
        $producto->update(['cantidad' => $producto->cantidad - $request->cantidad]);
    
        // Buscar si el producto existe en la sucursal de destino
        $productoDestino = Producto::where('codigo1', $producto->codigo1)
            ->where('ubicacion_id', $request->ubicacion_destino_id)
            ->first();
    
        if ($productoDestino) {
            // Si existe, aumentar la cantidad en la sucursal de destino
            $productoDestino->update([
                'cantidad' => $productoDestino->cantidad + $request->cantidad,
            ]);
        } else {
            // Crear una nueva instancia del producto en la sucursal de destino
            Producto::create([
                'nombre' => $producto->nombre,
                'codigo1' => $producto->codigo1,
                'codigo2' => $producto->codigo2,
                'descripcion' => $producto->descripcion,
                'precio_unidad_facturado' => $producto->precio_unidad_facturado,
                'precio_docena_facturado' => $producto->precio_docena_facturado,
                'precio_paquete_facturado' => $producto->precio_paquete_facturado,
                'precio_paquete_normal' => $producto->precio_paquete_normal,
                'precio_docena_normal' => $producto->precio_docena_normal,
                'precio_docena_caja' => $producto->precio_docena_caja,
                'piezas_por_paquete' => $producto->piezas_por_paquete,
                'piezas_por_caja' => $producto->piezas_por_caja,
                'ubicacion_deposito' => $producto->ubicacion_deposito,
                'descuento' => $producto->descuento,
                'incremento' => $producto->incremento,
                'marca' => $producto->marca,
                'foto' => $producto->foto,
                'cantidad' => $request->cantidad,
                'ubicacion_id' => $request->ubicacion_destino_id,
                'estado' => 'activo',
                'posicion' => $producto->posicion,
                'docena' => $producto->docena,
            ]);
        }
    
        // Registrar el traspaso
        Traspaso::create([
            'producto_id' => $producto->id,
            'ubicacion_origen_id' => $request->ubicacion_origen_id,
            'ubicacion_destino_id' => $request->ubicacion_destino_id,
            'cantidad' => $request->cantidad,
            'fecha' => now(),
            'user_id' => Auth::id(),
            'estado' => 'enviado',
        ]);
    
        return redirect()->route('traspasos.index')->with('success', 'Producto transferido exitosamente.');
    }
    
public function recibirProducto(Request $request)
{
    $configuracion = TraspasoConfig::first();

    $request->validate([
        'producto_id' => 'required|exists:productos,id',
        'ubicacion_origen_id' => 'required|exists:ubicaciones,id',
        'ubicacion_destino_id' => 'required|exists:ubicaciones,id|different:ubicacion_origen_id',
        'cantidad' => 'required|integer|min:1|max:' . $configuracion->cantidad_maxima,
    ], [
        'ubicacion_destino_id.different' => 'La Sucursal de Destino debe ser diferente a la Sucursal de Origen.',
        'cantidad.max' => 'La cantidad no puede ser mayor a ' . $configuracion->cantidad_maxima . '.',
        'cantidad.min' => 'La cantidad mínima para recibir debe ser de 1 producto.',
    ]);

    $producto = Producto::findOrFail($request->producto_id);

    // Verificar o crear producto en la ubicación de destino
    $ubicacionDestino = $producto->ubicaciones->where('id', $request->ubicacion_destino_id)->first();
    if ($ubicacionDestino) {
        // Si el producto ya existe en la sucursal de destino, sumamos la cantidad
        $ubicacionDestino->pivot->cantidad += $request->cantidad;
        $ubicacionDestino->pivot->save();
    } else {
        // Crear una nueva relación para la sucursal destino con la cantidad enviada
        $producto->ubicaciones()->attach($request->ubicacion_destino_id, ['cantidad' => $request->cantidad]);
    }

    // Registrar el traspaso
    $traspaso = new Traspaso([
        'producto_id' => $request->producto_id,
        'ubicacion_origen_id' => $request->ubicacion_origen_id,
        'ubicacion_destino_id' => $request->ubicacion_destino_id,
        'cantidad' => $request->cantidad,
        'fecha' => now(),
        'user_id' => Auth::id(),
        'estado' => 'recibido',
    ]);
    $traspaso->save();

    return redirect()->route('traspasos.index')->with('success', 'Producto recibido exitosamente.');
}

    public function showFormRecibirProducto()
    {
        $productos = Producto::all();
        $ubicaciones = Ubicacion::all();

        return view('traspasos.recibir_producto', compact('productos', 'ubicaciones'));
    }

    
    
}