<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use Illuminate\Http\Request;
use App\Models\Venta;
use App\Models\Producto;
use App\Models\TraspasoConfig;
use App\Models\Ubicacion;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade as FacadePdf;
use Illuminate\Support\Facades\View;
use Barryvdh\DomPDF\Facade as PDF;
use Barryvdh\DomPDF\Facade\Pdf as DomPDFFacadePdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use PhpOffice\PhpSpreadsheet\Writer\Pdf as WriterPdf;

class VentaController extends Controller
{
    
    
    public function index(Request $request)
    {
        $user = Auth::user();
    
        // Consulta base con relaciones
        $query = Venta::with('usuario', 'productosVendidos.producto.ubicacion', 'usuarioAprobador', 'cliente'); // Asegúrate de incluir la relación 'cliente'
    
        // Restricción por rol
        if ($user->hasRole('Empleado')) {
            $ubicacion_id = $user->empleado->ubicacion_id ?? null;
            if ($ubicacion_id) {
                $query->whereHas('productosVendidos.producto.ubicacion', function ($q) use ($ubicacion_id) {
                    $q->where('id', $ubicacion_id);
                });
            }
        }
    
        // Filtro por estado
        if ($request->filled('estado')) {
            $query->where('estado', $request->input('estado'));
        }
    
        // Filtro por cliente
        if ($request->filled('cliente')) {
            $cliente = $request->input('cliente');
            $query->whereHas('cliente', function ($q) use ($cliente) {
                $q->where('nombre', 'like', "%$cliente%")
                  ->orWhere('apellidos', 'like', "%$cliente%")
                  ->orWhere('carnet', 'like', "%$cliente%");
            });
        }
    
        // Filtro por user_id (usuario)
        if ($request->filled('user_id')) {
            $user_id = $request->input('user_id');
            $query->where('user_id', $user_id); // Asegúrate de que 'usuario_id' es el campo correcto en la tabla de ventas
        }
    
        // Filtros de búsqueda
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q->where('codigo', 'like', "%$search%")
                  ->orWhereHas('productosVendidos.producto', function ($q) use ($search) {
                      $q->where('nombre', 'like', "%$search%")
                        ->orWhere('codigo1', 'like', "%$search%")
                        ->orWhere('codigo2', 'like', "%$search%")
                        ->orWhere('descripcion', 'like', "%$search%");
                  });
            });
        }
    
        // Filtro por ubicación
        if ($request->filled('ubicacion')) {
            $ubicacion_id = $request->input('ubicacion');
            $query->whereHas('productosVendidos.producto.ubicacion', function ($q) use ($ubicacion_id) {
                $q->where('id', $ubicacion_id);
            });
        }
    
        // Filtro por usuario aprobador
        if ($request->filled('usuario_aprobador')) {
            $usuarioAprobadorId = $request->input('usuario_aprobador');
            $query->where('usuario_aprobador_id', $usuarioAprobadorId);
        }
    
        // Filtro por rango de fechas
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
            $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
            $query->whereBetween('fecha', [$startDate, $endDate]);
        }
    
        // Ordenar por la última venta
        $query->orderBy('fecha', 'desc')->orderBy('hora', 'desc');
    
        // Paginación de ventas
        $ventasPaginadas = $query->paginate(10)->appends($request->all());
    
        // Obtener todos los usuarios para el filtro de usuario aprobador
        $usuariosAprobadores = User::all();
    
        // Obtener todos los clientes para el filtro de cliente
        $clientes = Cliente::all(); // Cambia esto según la relación de clientes en tu aplicación
    
        // Obtener todas las ubicaciones para el filtro
        $ubicaciones = Ubicacion::all();
    
        return view('ventas.index', compact('ventasPaginadas', 'ubicaciones', 'usuariosAprobadores', 'clientes'));
    }
    
    
    
    
    
    public function aplicarTipoCambio(Request $request)
    {
        $request->validate([
            'tipo_cambio' => 'required|numeric|min:0',
        ]);
    
        session(['tipo_cambio' => $request->tipo_cambio]);
    
        return back()->with('success', 'Tipo de cambio aplicado correctamente.');
    }
    public function getProductosPorUbicacion($id)
{
    $productos = Producto::where('ubicacion_id', $id)->get();

    if ($productos->isEmpty()) {
        return response()->json([], 404);  // Retorna 404 si no hay productos
    }

    return response()->json($productos);
}

    public function eliminarTipoCambio()
    {
        session()->forget('tipo_cambio');
    
        return back()->with('success', 'Tipo de cambio reiniciado.');
    }
    
    public function generatePDF(Request $request)
    {
        $user = Auth::user();
        $query = Venta::with('usuario', 'productosVendidos.producto.ubicacion', 'usuarioAprobador'); // Asegúrate de que 'usuarioAprobador' esté relacionado.
        
        // Restricción por rol
        if ($user->hasRole('Empleado')) {
            $ubicacion_id = $user->empleado->ubicacion_id ?? null;
            if ($ubicacion_id) {
                $query->where('ubicacion_id', $ubicacion_id);
            }
        }
        
        // Filtro por búsqueda
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q->where('codigo', 'like', "%$search%") // Búsqueda por código de venta
                  ->orWhereHas('productosVendidos.producto', function ($q) use ($search) {
                      $q->where('nombre', 'like', "%$search%")
                        ->orWhere('codigo1', 'like', "%$search%")
                        ->orWhere('codigo2', 'like', "%$search%")
                        ->orWhere('descripcion', 'like', "%$search%");
                  });
            });
        }
    
        // Filtro por cliente (nuevo)
        if ($request->filled('cliente')) {
            $cliente = $request->input('cliente');
            $query->where(function ($q) use ($cliente) {
                $q->whereHas('cliente', function ($q) use ($cliente) {
                    $q->where('nombre', 'like', "%$cliente%")
                      ->orWhere('apellido', 'like', "%$cliente%")
                      ->orWhere('carnet', 'like', "%$cliente%");
                });
            });
        }
    
        // Filtro por usuario (nuevo)
        if ($request->filled('user_id')) {
            $user_id = $request->input('user_id');
            $query->where('user_id', $user_id);
        }
    
        // Filtro por rango de fechas
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $startDate = Carbon::parse($request->input('start_date'))->startOfDay();
            $endDate = Carbon::parse($request->input('end_date'))->endOfDay();
            $query->whereBetween('fecha', [$startDate, $endDate]);
        }
    
        // Filtro por estado
        if ($request->filled('estado')) {
            $estado = $request->input('estado');
            $query->where('estado', $estado);
        }
    
        // Filtro por ubicación
        if ($request->filled('ubicacion')) {
            $ubicacion_id = $request->input('ubicacion');
            $query->whereHas('productosVendidos.producto.ubicacion', function ($q) use ($ubicacion_id) {
                $q->where('id', $ubicacion_id);
            });
        }
    
        // Filtro por usuario aprobador
        if ($request->filled('usuario_aprobador')) {
            $usuarioAprobadorId = $request->input('usuario_aprobador');
            $query->where('usuario_aprobador_id', $usuarioAprobadorId);
        }
    
        // Obtener los datos para el reporte
        $ventas = $query->get();
    
        // Generar el PDF
        $view = View::make('ventas.pdf', compact('ventas'));
        $pdf = DomPDFFacadePdf::loadHTML($view->render());
    
        // Nombre del archivo PDF
        $pdfName = 'reporte_ventas_' . now()->format('Y_m_d_His');
        if ($request->filled('pdfName')) {
            $pdfName .= '_' . $request->input('pdfName');
        }
    
        return $pdf->download($pdfName . '.pdf');
    }
    
    

    public function generarResumenPDF($id)
    {
        $venta = Venta::with('productosVendidos.producto')->findOrFail($id);
    
    
    
        $view = view('ventas.resumen-pdf', compact('venta'))->render();
    
        $pdf = DomPDFFacadePdf::loadHTML($view);
        return $pdf->stream('resumen_venta_' . $venta->codigo . '.pdf'); // Stream para abrir en nueva pestaña
    }
 
    public function generarAlmacenPDF($id)
    {
        $venta = Venta::with('productosVendidos.producto')->findOrFail($id);
    
     
    
        $view = view('ventas.almacen-pdf', compact('venta'))->render();
    
        $pdf = DomPDFFacadePdf::loadHTML($view);
        $pdf->setPaper([0, 0, 226, 1000]); // 80 mm de ancho
        return $pdf->stream('almacen_venta_' . $venta->codigo . '.pdf'); 
    }
     
     
    public function generarSimplePDF($id)
    {
        $venta = Venta::with('productosVendidos.producto')->findOrFail($id);
    
    
    
        $view = view('ventas.simple-pdf', compact('venta'))->render();
    
        $pdf = DomPDFFacadePdf::loadHTML($view);
        $pdf->setPaper([0, 0, 226, 1000]); // 80 mm de ancho
        return $pdf->stream('simple_venta_' . $venta->codigo . '.pdf'); 
    }
    
    
    
    public function create()
{
    $user = Auth::user();

    // Obtener el tipo de cambio actual desde TraspasoConfig
    $tipoCambioConfig = TraspasoConfig::first();
    $tipoCambio = $tipoCambioConfig->tipo_cambio ?? 1; // Valor predeterminado: 1

    // Obtener los productos y ubicaciones según el rol del usuario
if ($user->hasRole('Super administrador')) {
    $productos = Producto::with('ubicacion')->get();
} elseif ($user->hasRole('Administrador')) {
    $productos = Producto::with('ubicacion')->get();
} elseif ($user->hasRole('Empleado')) {
    $ubicacion = $user->empleado->ubicacion_id;
    $productos = Producto::with('ubicacion')->where('ubicacion_id', $ubicacion)->get();
} else {
    return redirect()->back()->with('error', 'No tienes permisos para realizar esta acción.');
}


    $ubicaciones = Ubicacion::all();
    $clientes = Cliente::where('estado', 1)->get(); // Filtrar clientes con estado 1


    return view('ventas.create', compact('productos', 'ubicaciones', 'tipoCambio', 'clientes'));
}

    public function store(Request $request)
    {
        $request->validate([
            'productos' => 'required|string',
            'precio_total' => 'required|numeric|min:0',
            'cliente_id' => 'nullable|exists:clientes,id', // Validar si el cliente existe
            'observacion' => 'nullable|string', // Observacion puede ser nula
        ]);
    
        // Decodificar el JSON de productos
        $productos = json_decode($request->productos, true);
    
        if (!$productos || !is_array($productos)) {
            return redirect()->back()->withErrors('Los datos de productos son inválidos.');
        }
    
        try {
            // Iniciar transacción
            DB::transaction(function () use ($request, $productos) {
                // Crear la venta
                $venta = Venta::create([
                    'user_id' => Auth::id(),
                    'precio_total' => $request->precio_total,
                    'fecha' => now()->format('Y-m-d'),
                    'hora' => now()->format('H:i:s'),
                    'estado' => 0, // Estado inicial: Pendiente
                    'codigo' => Venta::generarCodigoUnico(), // Generar código único
                    'cliente_id' => $request->cliente_id, // Campo cliente_id
                    'observacion' => $request->observacion, // Campo observacion
                ]);
    
                foreach ($productos as $productoData) {
                    $producto = Producto::findOrFail($productoData['id']);
                
                    // Validar que la cantidad total enviada sea menor o igual al stock disponible
                    if ($producto->cantidad < $productoData['cantidad_total']) {
                        throw new \Exception("Stock insuficiente para el producto: {$producto->nombre}. Stock actual: {$producto->cantidad}, requerido: {$productoData['cantidad_total']}");
                    }
                
                    // Descontar la cantidad del inventario
                    $producto->decrement('cantidad', $productoData['cantidad_total']);
                
                    // Registrar el producto vendido
                    $venta->productosVendidos()->create([
                        'producto_id' => $producto->id,
                        'cantidad' => $productoData['cantidad'], // Cantidad en la medida seleccionada
                        'cantidad_total' => $productoData['cantidad_total'], // Cantidad en unidades reales
                        'tipo_medida' => $productoData['tipo_medida'], // Tipo de medida (docena, paquete, etc.)
                        'precio' => $productoData['precio'], // Precio original
                        'precio_total' => $productoData['precio_total'], // Total del producto
                        'tipo_precio' => $productoData['tipo_precio'], // Tipo de precio (PDF, PUF, etc.)
                    ]);
                }
                
            });
    
            // Redirigir con éxito
            return redirect()->route('ventas.index')->with('success', 'Venta registrada exitosamente.');
        } catch (\Exception $e) {
            // Capturar cualquier error durante la transacción y redirigir con un mensaje
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    
    
    
    public function aprobar($id)
    {
        try {
            $venta = Venta::findOrFail($id);
    
            if ($venta->estado === 1) {
                return response()->json(['error' => 'Esta venta ya está aprobada.'], 400);
            }
    
            $venta->update([
                'estado' => 1,
                'usuario_aprobador_id' => Auth::id(),
            ]);
    
            return response()->json(['success' => 'Venta aprobada exitosamente.']);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Error al aprobar la venta.'], 500);
        }
    }
    

    public function rechazar($id)
    {
        try {
            // Buscar la venta con las relaciones necesarias
            $venta = Venta::with('productosVendidos.producto')->findOrFail($id);
    
            // Verificar si la venta ya está rechazada o aprobada
            if ($venta->estado === 2) {
                return response()->json(['error' => 'Esta venta ya está rechazada.'], 400);
            }
            if ($venta->estado === 1) {
                return response()->json(['error' => 'Esta venta ya está aprobada. No se puede rechazar.'], 400);
            }
    
            // Iniciar transacción para rechazar la venta y devolver inventario
            DB::transaction(function () use ($venta) {
                // Iterar sobre los productos vendidos y devolverlos al inventario
                foreach ($venta->productosVendidos as $productoVendido) {
                    $producto = Producto::findOrFail($productoVendido->producto_id);
    
                    // Devolver la cantidad al inventario
                    $producto->increment('cantidad', $productoVendido->cantidad_total);
                }
    
                // Cambiar el estado de la venta a rechazada
                $venta->update([
                    'estado' => 2, // Estado de la venta rechazada
                    'usuario_rechazador_id' => Auth::id(), // El usuario que rechaza
                ]);
            });
    
            // Responder con un mensaje de éxito
            return response()->json(['success' => 'Venta rechazada y productos devueltos al inventario.']);
        } catch (\Exception $e) {
            // Manejo de excepciones con un mensaje claro
            return response()->json(['error' => 'Error al rechazar la venta: ' . $e->getMessage()], 500);
        }
    }
    
    
    
}